<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Rules\UniqueGroupName;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\JsonResponse;
use Illuminate\View\View;
use App\Models\Group;
use App\Http\Helper\Helper;
use Auth;
use DB;

class GroupController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(): View
    {   

        $page_variables = $this->_pageVariables(request('type'));
        Helper::checkPermissions($page_variables['page']);

        $page = $page_variables['page'];
        $page_title = __('app.groups');
        $breadcrumbs = [
            __('app.groups') => '#',
            $page_variables['breadcrumbs_title'] => route('groups.index', ['type' => request('type')])
        ];  
        return view('groups.index',compact('page', 'page_title', 'breadcrumbs'));
    }

    /**
       * Retrun JSON datatable data
    */
    public function getGroupLists(Request $request): Void
    {
        $result = Group::select('id', 'name', 'created_at')
            ->where('type_id', $request->type_id)
            ->app();

        $columns = ['id', 'id', 'name', 'created_at'];

        $data = Helper::dataFilters($request, $result, $columns);

        $result = $data['result'];
        $groups = $result->get();

        $data =  Helper::datatableTotals($data['total']);

        foreach($groups as $group) {
          $checkbox = "<input type=\"checkbox\" value=\"{$group->id}\" class=\"form-check-input\">";

          $actions = '<div class="btn-group">';
          $actions .= '<button type="button" class="btn btn-outline-primary dropdown-toggle" data-bs-toggle="dropdown" aria-expanded="false">'.__('app.actions').'</button>';
          $actions .= '<ul class="dropdown-menu" style="">';
          $actions.= '<li><a class="dropdown-item" href="javascript:;" onclick="viewModal(\'modal\', \''.route('groups.edit', $group->id).'\')"><i class="bi bi-pencil"></i> '.__('app.edit').'</a></li>';


          $actions .= '<li><a class="dropdown-item text-danger" href="javascript:;" onclick="destroy(\''.$group->id.'\', \''.route('groups.destroy', [$group->id, "type=$request->type_id"]).'\')"><i class="bi bi-trash"></i> '.__('app.delete').'</a></li>';
          $actions .= '</ul>';

          $data['data'][] = [
            "DT_RowId" => "row_{$group->id}",
            $checkbox,
            $group->id,
            $group->name,
            Helper::datetimeDisplay($group->created_at),
            $actions
          ];
        }
        echo json_encode($data);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(int $type_id)
    {
        $group_type = $this->_pageVariables($type_id);
        Helper::checkPermissions($group_type['page']);
        return view('groups.create', compact('type_id', 'group_type'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request): RedirectResponse
    {
        $request->validate([
          'name' => ['required', 'string', new UniqueGroupName],
        ]);
    
        $input = $request->all();
        $input['app_id'] = Auth::user()->app_id;
        $input['user_id'] = Auth::user()->id;
        $input['type_id'] = $request->group_type;
    
        $user = Group::create($input);

        // save log
        activity('create')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.group') . " ({$request->name}) ". __('app.log_create')); // log

        return redirect()->back()
            ->with('success', __('app.group') . ' ' . __('app.log_create'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Group $group): View
    {
        $group_type = $this->_pageVariables($group->type_id);
        Helper::checkPermissions($group_type['page']);
        return view('groups.edit', compact('group', 'group_type'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Group $group): RedirectResponse
    {
        $request->validate([
          'name' => ['required', 'string', new UniqueGroupName],
        ]);

        $input = $request->all();
        $group->update($input);

        // save log
        activity('update')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.group') . " ({$request->name}) ". __('app.log_update')); // log

        // redirect
        return redirect()->back()
            ->with('success', __('app.group') . ' ' . __('app.updated_successfully'));

    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(int $id, Request $request): JsonResponse
    {
        if($request->type == config('custom.groups_lists'))
            $exists = DB::table('lists')->where('group_id', $id)->exists();
        elseif($request->type == config('custom.group_sending_servers'))
            $exists = DB::table('sending_servers')->where('group_id', $id)->exists();
        elseif($request->type == config('custom.group_sending_domains'))
            $exists = DB::table('sending_domains')->where('group_id', $id)->exists();
        elseif($request->type == config('custom.group_campaigns'))
            $exists = DB::table('broadcasts')->where('group_id', $id)->exists();

        if ($exists) {
            $data = [
                'success' => false,
                'message' => __('app.group_not_deleted_list_associated')
            ];            
        } else {
            if(!empty($request->action)) {
              $ids = array_values($request->ids);
              $names = json_encode(array_values(Group::whereIn('id', $ids)->pluck('name')->toArray()));
              $destroy = Group::whereIn('id', $ids)->delete();
            } else {
              $names = Group::whereId($id)->value('name');
              $destroy = Group::destroy($id);
            }

            activity('delete')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.group') . " ({$names}) ". __('app.log_delete'));

            $data = [
                'success' => true,
                'message' => __('app.deleted_successfully')
            ];
        }
        return response()->json($data, 200); 
    }

    /**
     * Return an array of page variable accordingly
     */
    private function _pageVariables ($type): Array
    {
        if($type == config('custom.group_campaigns')) {
            $page_variables['page'] = 'groups_groupcampaigns';
            $page_variables['breadcrumbs_title'] = __('app.campaigns');
        } elseif($type == config('custom.group_sending_servers')) {
            $page_variables['page'] = 'groups_sending_groupservers';
            $page_variables['breadcrumbs_title'] = __('app.delivery_servers');
        } elseif($type == config('custom.group_sending_domains')) {
            $page_variables['page'] = 'groups_sending_groupdomains';
            $page_variables['breadcrumbs_title'] = __('app.sending_domains');
        } elseif($type == config('custom.group_suppression')) {
            $page_variables['page'] = 'groups_groupsuppression';
            $page_variables['breadcrumbs_title'] = __('app.suppression');
        } else {
            $page_variables['page'] = 'groups_grouplists';
            $page_variables['breadcrumbs_title'] = __('app.lists');
        }

        return $page_variables;
    }

    /**
   * Delete or Erease group
  */
  public function deleteGroup($model, Request $request)
  {
    $id = $request->id;
    $action = $request->action;
    $model = $request->model;

    if($model == 'list') {
      if($action == 'erase') {
        $destroy = \App\Models\Lists::whereGroupId($id)->delete();
      } elseif($action == 'move') {
        if(!empty($request->move_id))
          $destroy = \App\Models\Lists::whereId($request->move_id)->update(['group_id' => $request->group_id_new]);
        else
          $destroy = \App\Models\Lists::whereGroupId($id)->update(['group_id' => $request->group_id_new]);
      }
    } elseif($model == 'broadcast') {
      if($action == 'erase') {
        $destroy = \App\Models\Broadcast::whereGroupId($id)->delete();
      } elseif($action == 'move') {
        if(!empty($request->move_id))
          $destroy = \App\Models\Broadcast::whereId($request->move_id)->update(['group_id' => $request->group_id_new]);
        else
          $destroy = \App\Models\Broadcast::whereGroupId($id)->update(['group_id' => $request->group_id_new]);
      }
    } elseif($model == 'sending_server') {
      if($action == 'erase') {
        $destroy = \App\Models\SendingServer::whereGroupId($id)->delete();
      } elseif($action == 'move') {
        if(!empty($request->move_id))
          $destroy = \App\Models\SendingServer::whereId($request->move_id)->update(['group_id' => $request->group_id_new]);
        else
          $destroy = \App\Models\SendingServer::whereGroupId($id)->update(['group_id' => $request->group_id_new]);
      }
    } elseif($model == 'sending_domain') {
      if($action == 'erase') {
        $destroy = \App\Models\SendingDomain::whereGroupId($id)->delete();
      } elseif($action == 'move') {
        if(!empty($request->move_id))
          $destroy = \App\Models\SendingDomain::whereId($request->move_id)->update(['group_id' => $request->group_id_new]);
        else
          $destroy = \App\Models\SendingDomain::whereGroupId($id)->update(['group_id' => $request->group_id_new]);
      }
    } elseif($model == 'suppression') {
      if($action == 'erase') {
        $destroy = \App\Models\Suppression::whereGroupId($id)->delete();
      } elseif($action == 'move') {
        if(!empty($request->move_id))
          $destroy = \App\Models\Suppression::whereId($request->move_id)->update(['group_id' => $request->group_id_new]);
        else
          $destroy = \App\Models\Suppression::whereGroupId($id)->update(['group_id' => $request->group_id_new]);
      }
    }

    if(!empty($id)) {
      $group_name = Group::whereId($id)->value('name');
      activity('delete')->withProperties(['app_id' => Auth::user()->app_id])->log(__('app.group') . " ({$group_name}) ". __('app.log_delete')); // log
      Group::destroy($id);
    }
    return $destroy;
  }
}
