<?php

namespace App\Jobs;

use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use App\Http\Helper\Helper;
use League\Csv\Writer;

class ListExport implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;
    public $tries = 1;
    protected $list_id, $user_id, $fields;

    /**
     * Create a new job instance.
     */
    public function __construct($list_id, $user_id, $fields)
    {
        $this->list_id = $list_id;
        $this->user_id = $user_id;
        $this->fields = $fields;
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        $list = \App\Models\Lists::findOrFail($this->list_id);
        $path_export_list = str_replace('[user-id]', $this->user_id, config('custom.path_export_list'));
        Helper::dirCreateOrExist($path_export_list); // create dir if not exist

        $file = $path_export_list.\Illuminate\Support\Str::slug($list->name).'-'.Helper::uniqueID().'.csv';
        $writer = Writer::createFromPath($file, 'w+'); // create a .csv file to write data

        // create file header
        $list_custom_fields = $list->customFields()->orderBy('custom_fields.id')->pluck('name');

        $file_header = [
          __('app.contact_email'),
          __('app.active'),
          __('app.unsubscribed'),
          __('app.confirmed'),
          __('app.verified'),          
        ];
        foreach($list_custom_fields as $custom_field) {
          array_push($file_header, $custom_field);
        }

        $writer->insertOne($file_header); // write file header

        $list_contacts = \App\Models\Contact::whereListId($this->list_id)
          ->with('customFields');

        if($this->fields['is_active'] == 'yes') {
            $list_contacts->where('is_active', true);
        } elseif($this->fields['is_active'] == 'no') {
            $list_contacts->where('is_active', false);
        }

        if($this->fields['is_confirmed'] == 'yes') {
            $list_contacts->where('is_confirmed', true);
        } elseif($this->fields['is_confirmed'] == 'no') {
            $list_contacts->where('is_confirmed', false);
        }

        if($this->fields['is_verified'] == 'yes') {
            $list_contacts->where('is_verified', true);
        } elseif($this->fields['is_verified'] == 'no') {
            $list_contacts->where('is_verified', false);
        }

        if($this->fields['is_subscribed'] == 'yes') {
            $list_contacts->where('is_unsubscribed', false);
        } elseif($this->fields['is_subscribed'] == 'no') {
            $list_contacts->where('is_unsubscribed', true);
        }

        $list_contacts = $list_contacts->chunk(1000, function ($contacts)  use ($writer) {
            foreach($contacts as $contact) {
              $contact_data = [
                $contact->email,
                $contact->is_active ? 1 : 0,
                $contact->is_unsubscribed ? 0 : 1,
                $contact->is_confirmed ? 1 : 0,
                $contact->is_verified ? 1 : 0,                
              ];

              $custom_fields = $contact->customFields()->orderBy('custom_fields.id')->get();
              if(!empty($custom_fields)) {
                foreach($custom_fields as $custom_field) {
                  $custom_field_data = str_replace('||', ', ', $custom_field->pivot->data);
                  array_push($contact_data, $custom_field_data);
                }
              }

              $writer->insertOne($contact_data); // write contact info
            }
        });

        // save notification for user to inform and download
        $notification_name = __('app.list'). " ({$list->name}) "  . __('app.msg_export_successfully');
        $attributes = [
          'file' => $file
        ];
        $notification_attributes = json_encode($attributes);
        $notification = [
          'name' => $notification_name,
          'type' => 'export',
          'attributes' => $notification_attributes,
          'app_id' => $list->app_id,
          'user_id' => $this->user_id
        ];
        \App\Models\Notification::create($notification);
    }
}
