<?php

namespace App\Jobs;

use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use App\Models\GlobalBounce;
use League\Csv\Writer;
use App\Http\Helper\Helper;

class BouncedExport implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected $app_id, $user_id;
    public $tries = 1;

    /**
     * Create a new job instance.
     */
    public function __construct($app_id, $user_id)
    {
        $this->app_id = $app_id;
        $this->user_id = $user_id;
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        $path_export_blacklist = str_replace('[user-id]', $this->user_id, config('custom.path_export_blacklist'));
        Helper::dirCreateOrExist($path_export_blacklist); // create dir if not exist

        $file = $path_export_blacklist. 'bounces-'.Helper::uniqueID().'.csv';
        $writer = Writer::createFromPath($file, 'w+'); // create a .csv file to write data

        $file_header = [
          __('app.email'),
          __('app.type'),
          __('app.code'),
          __('app.detail'),
          __('app.datetime'),
        ];

        $writer->insertOne($file_header); // write file header

        \App\Models\GlobalBounce::whereNotNull('id')
         ->chunk(1000, function ($bounces)  use ($writer) {
            $timezone = \App\Models\User::getUserValue($this->user_id, 'timezone');
            foreach($bounces as $bounce) {
              $bounce_data = [
                $bounce->email,
                $bounce->type,
                $bounce->code,
                json_decode($bounce->detail)->short_detail ?? '',
                Helper::datetimeDisplay($bounce->created_at, $timezone),
              ];

              $writer->insertOne($bounce_data); // write contact info
            }
         });

         // save notification for user to inform and download
        $notification_name = __('app.system_bounced') .' ' . __('app.msg_export_successfully');
        $attributes = [
          'file' => $file
        ];
        $notification_attributes = json_encode($attributes);
        $notification = [
          'name' => $notification_name,
          'type' => 'export',
          'attributes' => $notification_attributes,
          'app_id' => $this->app_id,
          'user_id' => $this->user_id
        ];
        \App\Models\Notification::create($notification);
    }
}
